#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <stdbool.h>
#include <omp.h>


/*
 * structure for keeping the algorithm state and relevant functions regarding the state
 * */
 
#define MAX_NUM_NODES 1000
#define INVALID_NODE -1

struct State {
    int cliqueSize, nodesLeft, graphSize, numBranches;
    short clique[MAX_NUM_NODES];
    short nodesToCheck[MAX_NUM_NODES];
};


void initState(struct State* state, int graphSize) {
    if (state == 0)
        return;
    state->nodesLeft = graphSize;
    state->graphSize = graphSize;
    state->cliqueSize = 0;
    state->numBranches = 0;
    for (int i = 0; i < MAX_NUM_NODES; ++i) {
        state->clique[i] = INVALID_NODE;
        state->nodesToCheck[i] = (i < graphSize ? i : INVALID_NODE);
    }
}

void copyState(struct State* from, struct State* to) {
    to->cliqueSize = from->cliqueSize;
    to->nodesLeft = from->nodesLeft;
    to->graphSize = from->graphSize;
    to->numBranches = 0;
    
    for (int i = 0; i<from->graphSize; ++i) {
        to->clique[i] = from->clique[i];
        to->nodesToCheck[i] = from->nodesToCheck[i];
    }
}

/*
 * Definition of a graph and functions dealing with graph (general and unrelated to clique)
 * */
 
 struct Graph {
     short numNodes;
     bool edges[MAX_NUM_NODES][MAX_NUM_NODES];
 };
 
 struct Graph* generateRandomGraph(int graphSize, float connectivityFactor) {
     if ((graphSize > MAX_NUM_NODES) || (connectivityFactor >= 1))
        return 0;
        
     struct Graph* g = malloc(sizeof(struct Graph));
     g->numNodes = graphSize;
     int maxEdges = (graphSize * (graphSize-1)) / 2;
     int wantEdges = (int)(maxEdges * connectivityFactor);
     bool negateGraph = false;
     
     for (int i = 0; i < graphSize; ++i) {
         for (int j = 0; j < graphSize; ++j) {
             g->edges[i][j] = false;
         }
     }
     
     if (maxEdges - wantEdges < wantEdges) {
         // negate graph to ease initializing edges
         wantEdges = maxEdges - wantEdges;
         negateGraph = true;
     }
     
     for (int i = 0; i < wantEdges;) {
         short v1 = rand() % graphSize;
         short v2 = (v1 + 1 + (rand() % (graphSize-1))) % graphSize;
         if (g->edges[v1][v2] == false) {
             ++i;
             g->edges[v1][v2] = true;
             g->edges[v2][v1] = true;
         }
     }
     
     if (negateGraph) {         
         for (int i = 0; i < graphSize; ++i) {
             for (int j = 0; j < graphSize; ++j) {
                 g->edges[i][j] = (i == j ? false : !g->edges[i][j]);
             }
         }
     }
     
     return g;
 }
 
/*
 * Branch and bound maximum clique algorithm (very basic implementation)
 * */

void keepOnlyNeighbours(short node, struct State* state, struct Graph* g) {
    int maxNodeNum = state->nodesLeft-1;
    for (int i = maxNodeNum; i >= 0; --i) {
        if (!(g->edges[node][state->nodesToCheck[i]])) {
            // i-th node left is not a neighbour of node
            state->nodesToCheck[i] = state->nodesToCheck[maxNodeNum];
            maxNodeNum--;
        }
    }
    state->nodesLeft = maxNodeNum+1;
}

// return true if the branch can be eliminated (bound condition is sattisfied)
bool bound(struct Graph* g, struct State* state, int* maxCliqueSize) {
    // note that this is a basic, not very smart bound
    return ((state->nodesLeft + state->cliqueSize) <= *maxCliqueSize);
}

void branch(struct Graph* g, struct State* state, int* maxCliqueSize) {
	state->numBranches++;
    while (!(bound(g, state, maxCliqueSize))) {
        // branch on this node:
        state->nodesLeft--;
        short node = state->nodesToCheck[state->nodesLeft];

		// make copy of state to ease backtracking
		struct State* stateCopy = malloc(sizeof(struct State));
		copyState(state, stateCopy);

		// go down the branch with the node added to clique
		state->clique[state->cliqueSize] = node;
		state->cliqueSize++;
		
		if (*maxCliqueSize < state->cliqueSize) {
			#pragma omp critical
			{
				if (*maxCliqueSize < state->cliqueSize) // another check (maxCliqueSize could be modified on multithreaded environment in the meantime)
					*maxCliqueSize = state->cliqueSize;
			}
		}
		
		keepOnlyNeighbours(node, state, g);
			branch(g, state, maxCliqueSize);
		if (stateCopy->nodesLeft > 0) {
			branch(g, stateCopy, maxCliqueSize);
			state->numBranches +=  stateCopy->numBranches;
		}
		
		free(stateCopy);
    }
}
 
int maxClique(struct Graph* g, struct State* state) {
    initState(state, g->numNodes);
    int maxCliqueSize = 0;

    branch(g, state, &maxCliqueSize);
    return maxCliqueSize;
} 
 
/*
 * OpenMP parallel branch-and-bound
 * */

int maxCliquePara(struct Graph* g, struct State* state) {
    initState(state, g->numNodes);
    // maximum clique size becomes a global variable
    int maxCliqueSize = 0;
    
    // change one level of recursion into a loop
    #pragma omp parallel 
    {
        #pragma omp single
        {
			printf("   running on %d threads\n   performed this many branches: ", omp_get_num_threads());
            struct State s;
            for (int i = 0; state->nodesLeft>0; ++i) {
                state->nodesLeft--;
                short node = state->nodesToCheck[state->nodesLeft];

                // make copy of state to ease backtracking
                copyState(state, &s);

                // go down the branch with the node added to clique
                state->clique[state->cliqueSize] = node;
                state->cliqueSize++;
                
                keepOnlyNeighbours(node, state, g);
                #pragma omp task firstprivate(s)
                {
					branch(g, &s, &maxCliqueSize); 
					printf("%d ", s.numBranches);
					#pragma omp atomic
					state->numBranches += s.numBranches;
				}
            }
        }
        
		if (maxCliqueSize < state->cliqueSize) {
			#pragma omp critical
			{
				if (maxCliqueSize < state->cliqueSize) // another check (maxCliqueSize could be modified on multithreaded environment in the meantime)
					maxCliqueSize = state->cliqueSize;
			}
		}
        #pragma omp taskwait 
        #pragma omp single
        printf("\n");
    }

    return maxCliqueSize;
} 
 
/*
 * various functions and main
 * */
 
float seconds(time_t t1, time_t t2) {
    return (t2-t1)/(float)CLOCKS_PER_SEC;
}

int main(int argc, char** argv) {
	int graphSize;
	if (argc > 1)
        graphSize = atoi(argv[1]);
    else
        graphSize = 70;
        
    int randSeed;
	if (argc > 2)
        randSeed = atoi(argv[2]);
    else
        randSeed = 10;
        
    srand(randSeed);
    
    time_t timeStartGraphInit = clock();
    struct Graph* graph = generateRandomGraph(graphSize, 0.93f);
    printf("Graph initialization took %f s\n", seconds(timeStartGraphInit, clock()));
    {
        time_t timeStartBandB = clock();
        struct State state;
        int maxSize = maxClique(graph, &state);
        printf("Branch-and-bound took          %f s and found clique of size %d in %d branches\n", seconds(timeStartBandB, clock()), maxSize, state.numBranches);
    }
    {
        time_t timeStartBandB = clock();
        double ompTimeStart = omp_get_wtime();
        struct State state;
        int maxSize = maxCliquePara(graph, &state);
        printf("Parallel Branch-and-bound took %f s and found clique of size %d in %d branches\n", omp_get_wtime()-ompTimeStart, maxSize, state.numBranches);
        printf("In total, threads used %f s of processor time\n", seconds(timeStartBandB, clock()));
    }
}
